/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * @private
 * @module nmodule/alarm/rc/console/commands/ForceClearCommand
 */
define([
  'baja!',
  'lex!alarm',
  'Promise',
  'underscore',
  'dialogs',
  'bajaux/commands/Command',
  'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn'
], function (baja,
             lexs,
             Promise,
             _,
             dialogs,
             Command,
             alarmConsoleCommandMixIn) {
  'use strict';

  var lex              = lexs[0],
      ERROR_MSG_PREFIX = lex.get('alarm.console.acknowledge.insufficient.permissions');


  function getSelectedSubjects(table) {
    return _.invoke(table.getSelectedRows(), 'getSubject');
  }

  function makeForceClearParams(subjects, singleSource) {
    var params = {};
    if (singleSource) {
      params.ids = _(subjects).map(function (subject) {
        return subject.uuid;
      });
    }
    else {
      params.srcs = _(subjects).map(function (subject) {
        return subject.source;
      });
    }
    return params;
  }

  function reportError(failureList) {
    if (_.isArray(failureList) && failureList.length) {
      var errorMsg = ERROR_MSG_PREFIX + "<br/><br/>" + failureList.join("<br/>");
      dialogs.showOk(errorMsg);
    }
    return failureList;
  }

  function forceClear(view) {
    var viewModel = view.$getViewModel(),
        ss        = viewModel.$isSingleSourceModel(),
        table     = view.$getAlarmTable(),
        subjects  = getSelectedSubjects(table),
        params    = makeForceClearParams(subjects, ss);

    return Promise.resolve(subjects.length && viewModel.forceClearAlarms(params).then(reportError));
  }

  /**
   * Force clears the selected alarms. The alarms will be forcibly removed from the alarm database.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var ForceClearCommand = function (view) {
    Command.apply(this, [{
      module: 'alarm',
      lex: 'commands.forceClear',
      func: function () {


        baja.rpc({
                typeSpec: 'sbc:SbcUtilRpc',
                method: 'checkForceClear',
                args:[{records:view.getSelectedAlarmRecords()}]
                }
                )
                .then(function (returnValue) {
              if(returnValue== '' || returnValue== ""){
                     return dialogs.showYesNo(lex.get('commands.forceClear.check'))
          .yes(function () {
            return forceClear(view);
          })
          .promise();
          }
          else
          {
                return dialogs.showOk(returnValue);

          }

            });
      }
    }]);

    alarmConsoleCommandMixIn(this);

    this.$view = view;
    this.$showInMenu = true;
  };

  ForceClearCommand.prototype = Object.create(Command.prototype);
  ForceClearCommand.prototype.constructor = ForceClearCommand;

  ForceClearCommand.prototype.selectionChanged = function () {
    this.setEnabled(!this.$view.isSelectionEmpty());
  };

  return ForceClearCommand;
});
